<?php
/**
 * Widget API: Erado_Recent_Posts class
 */

class Erado_Recent_Posts extends WP_Widget {

	/**
	 * Sets up a new Recent Posts widget instance.
	 *
	 * @since 2.8.0
	 * @access public
	 */
	public function __construct() {
		$widget_ops = array(
			'classname'                   => 'widget_recent_post_with_image',
			'description'                 => __( 'Your site&#8217;s most recent Posts.', 'erado' ),
			'customize_selective_refresh' => true,
		);
		parent::__construct( 'recent-posts-with-image', __( 'JMS - Recent Posts With Image', 'erado' ), $widget_ops );
		$this->alt_option_name = 'widget_recent_post_with_image';
	}

	/**
	 * Outputs the content for the current Recent Posts widget instance.
	 *
	 * @since 2.8.0
	 * @access public
	 *
	 * @param array $args     Display arguments including 'before_title', 'after_title',
	 *                        'before_widget', and 'after_widget'.
	 * @param array $instance Settings for the current Recent Posts widget instance.
	 */
	public function widget( $args, $instance ) {
		if ( ! isset( $args['widget_id'] ) ) {
			$args['widget_id'] = $this->id;
		}

		$title = ( ! empty( $instance['title'] ) ) ? esc_attr($instance['title']) : 'Recent Posts';

		/** This filter is documented in wp-includes/widgets/class-wp-widget-pages.php */
		$title = apply_filters( 'widget_title', $title, $instance, $this->id_base );
		$number = ( ! empty( $instance['number'] ) ) ? absint( $instance['number'] ) : 5;
		if ( ! $number )
			$number = 5;
		$show_date = isset( $instance['show_date'] ) ? $instance['show_date'] : false;
        $show_comment = isset( $instance['show_comment'] ) ? $instance['show_comment'] : false;

		/**
		 * Filters the arguments for the Recent Posts widget.
		 *
		 * @since 3.4.0
		 *
		 * @see WP_Query::get_posts()
		 *
		 * @param array $args An array of arguments used to retrieve the recent posts.
		 */
		$r = new WP_Query( apply_filters( 'widget_posts_args', array(
			'posts_per_page'      => $number,
			'no_found_rows'       => true,
			'post_status'         => 'publish',
			'ignore_sticky_posts' => true
		) ) );

		if ($r->have_posts()) :
		?>
		<?php echo $args['before_widget']; ?>
		<?php if ( $title ) {
			echo $args['before_title'] . $title . $args['after_title'];
		} ?>
		<ul>
		<?php while ( $r->have_posts() ) : $r->the_post(); ?>
			<li class="clearfix">
                <div class="post-thumbnail pr">
        			<a href="<?php the_permalink(); ?>">
        			<?php if ( has_post_thumbnail() ) {
        				$img = wp_get_attachment_image_src( get_post_thumbnail_id( get_the_ID() ), 'full' );
        				$image = aq_resize( $img[0], 85, 85, true );
        				echo '<img src="' . esc_url( $image ) . '" width="85" height="85" alt="' . get_the_title() . '" />';
        			} else {
        				echo '<img src="' . ERADO_URL . '/assets/images/placeholder/square.png" width="85" height="85" alt="' . get_the_title() . '" />';
        			}
        			?>
        			</a>
        		</div>
				<div class="post-info">
					<h4 class="entry-title"><a href="<?php the_permalink(); ?>"><?php get_the_title() ? the_title() : the_ID(); ?></a></h4>
                    <div class="entry-meta">
                        <?php if ( $show_date ) : ?>
    						<span class="time updated"><i class="fa fa-calendar-o"></i><?php echo esc_html( get_the_date() ); ?></span>
    					<?php endif; ?>
                        <?php if ( $show_comment ) : ?>
                            <span class="comment-number"><i class="fa fa-comment-o"></i><?php echo sprintf( '<a href="%2$s">' . esc_html__( '%1$s Comment', get_comments_number(), 'erado' ) . '</a>', number_format_i18n( get_comments_number() ), get_comments_link() ); ?></span>
                        <?php endif; ?>
                    </div>
				</div>
			</li>
		<?php endwhile;?>

		</ul>
		<?php echo $args['after_widget']; ?>
		<?php
		// Reset the global $the_post as this query will have stomped on it
		wp_reset_postdata();

		endif;
	}

	/**
	 * Handles updating the settings for the current Recent Posts widget instance.
	 *
	 * @since 2.8.0
	 * @access public
	 *
	 * @param array $new_instance New settings for this instance as input by the user via
	 *                            WP_Widget::form().
	 * @param array $old_instance Old settings for this instance.
	 * @return array Updated settings to save.
	 */
	public function update( $new_instance, $old_instance ) {
		$instance = $old_instance;
		$instance['title'] = sanitize_text_field( $new_instance['title'] );
		$instance['number'] = (int) $new_instance['number'];
		$instance['show_date'] = isset( $new_instance['show_date'] ) ? (bool) $new_instance['show_date'] : false;
        $instance['show_comment'] = isset( $new_instance['show_comment'] ) ? (bool) $new_instance['show_comment'] : false;
		return $instance;
	}

	/**
	 * Outputs the settings form for the Recent Posts widget.
	 *
	 * @since 2.8.0
	 * @access public
	 *
	 * @param array $instance Current settings.
	 */
	public function form( $instance ) {
		$title     = isset( $instance['title'] ) ? esc_attr( $instance['title'] ) : '';
		$number    = isset( $instance['number'] ) ? absint( $instance['number'] ) : 5;
		$show_date = isset( $instance['show_date'] ) ? (bool) $instance['show_date'] : false;
        $show_comment = isset( $instance['show_comment'] ) ? (bool) $instance['show_comment'] : false;

?>
		<p>
			<label for="<?php echo $this->get_field_id( 'title' ); ?>"><?php _e( 'Title:', 'erado' ); ?></label>
			<input class="widefat" id="<?php echo $this->get_field_id( 'title' ); ?>" name="<?php echo $this->get_field_name( 'title' ); ?>" type="text" value="<?php echo $title; ?>" />
		</p>
		<p>
			<label for="<?php echo $this->get_field_id( 'number' ); ?>"><?php _e( 'Number of posts to show:', 'erado' ); ?></label>
			<input class="tiny-text" id="<?php echo $this->get_field_id( 'number' ); ?>" name="<?php echo $this->get_field_name( 'number' ); ?>" type="number" step="1" min="1" value="<?php echo $number; ?>" size="3" />
		</p>
		<p>
			<input class="checkbox" type="checkbox"<?php checked( $show_date ); ?> id="<?php echo $this->get_field_id( 'show_date' ); ?>" name="<?php echo $this->get_field_name( 'show_date' ); ?>" />
			<label for="<?php echo $this->get_field_id( 'show_date' ); ?>"><?php _e( 'Display post date?', 'erado' ); ?></label>
		</p>
        <p>
			<input class="checkbox" type="checkbox"<?php checked( $show_comment ); ?> id="<?php echo $this->get_field_id( 'show_comment' ); ?>" name="<?php echo $this->get_field_name( 'show_comment' ); ?>" />
			<label for="<?php echo $this->get_field_id( 'show_comment' ); ?>"><?php _e( 'Display post comment?', 'erado' ); ?></label>
		</p>
<?php
	}
}

function erado_register_widget_recent_posts() {
	register_widget( 'Erado_Recent_Posts' );
}
add_action( 'widgets_init', 'erado_register_widget_recent_posts' );

?>
